'use client';

import {
  Card,
  CardDescription,
  CardHeader,
  CardTitle,
  CardContent,
} from '@/components/ui/card';
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from '@/components/ui/carousel';
import { Zap, Fuel, Landmark, Mountain, Wrench, CircuitBoard, Cpu, Scaling, Shield, Wind, ShieldCheck, ClipboardCheck, RotateCw, ArrowRight, Sparkles, TrendingUp, Layers } from 'lucide-react';
import { useLanguage } from '@/contexts/language-context';
import Image from 'next/image';
import Link from 'next/link';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import { motion, AnimatePresence } from 'framer-motion';
import { useState } from 'react';

const ubcIconMap: { [key: string]: React.ElementType } = {
  'Energy & Solar Projects': Zap,
  'Oil & Gas Development': Fuel,
  'Project Finance & Guarantees': Landmark,
  'Mining & Metals': Mountain,
  'EPC & Lifecycle Management': RotateCw,
};

const noorIconMap: { [key: string]: React.ElementType } = {
  'Electrical, Instrumentation & Automation': CircuitBoard,
  'PLC, SCADA, DCS Systems': Cpu,
  'Calibration & Control Systems': Scaling,
  'Cathodic Protection Systems': Shield,
  'Renewable & Solar Energy Design': Wind,
  'IT, CCTV & Cybersecurity': ShieldCheck,
  'Commissioning & Maintenance': ClipboardCheck,
};

const colorVariants = [
  'from-blue-500/30 via-cyan-500/30 to-teal-500/30',
  'from-purple-500/30 via-pink-500/30 to-rose-500/30',
  'from-orange-500/30 via-amber-500/30 to-yellow-500/30',
  'from-green-500/30 via-emerald-500/30 to-lime-500/30',
  'from-indigo-500/30 via-violet-500/30 to-purple-500/30',
];

const noorServiceColors = [
  { gradient: 'from-blue-500 via-cyan-500 to-teal-400', bgGradient: 'from-blue-50 via-cyan-50 to-teal-50', accent: 'text-blue-600', border: 'border-blue-300/50' },
  { gradient: 'from-purple-500 via-pink-500 to-rose-400', bgGradient: 'from-purple-50 via-pink-50 to-rose-50', accent: 'text-purple-600', border: 'border-purple-300/50' },
  { gradient: 'from-emerald-500 via-green-500 to-lime-400', bgGradient: 'from-emerald-50 via-green-50 to-lime-50', accent: 'text-emerald-600', border: 'border-emerald-300/50' },
  { gradient: 'from-orange-500 via-amber-500 to-yellow-400', bgGradient: 'from-orange-50 via-amber-50 to-yellow-50', accent: 'text-orange-600', border: 'border-orange-300/50' },
  { gradient: 'from-red-500 via-rose-500 to-pink-400', bgGradient: 'from-red-50 via-rose-50 to-pink-50', accent: 'text-red-600', border: 'border-red-300/50' },
  { gradient: 'from-indigo-500 via-violet-500 to-fuchsia-400', bgGradient: 'from-indigo-50 via-violet-50 to-fuchsia-50', accent: 'text-indigo-600', border: 'border-indigo-300/50' },
  { gradient: 'from-gray-500 via-slate-500 to-stone-400', bgGradient: 'from-gray-50 via-slate-50 to-stone-50', accent: 'text-gray-600', border: 'border-gray-300/50' },
];


function ServiceCarousel({ title, items, badge, learnMore }: { title: string; items: { title: string; description: string }[]; badge: string; learnMore: string }) {
  const ubcImageMap = {
    'Energy & Solar Projects': PlaceHolderImages.find((img) => img.id === 'ubc-energy-solar'),
    'Oil & Gas Development': PlaceHolderImages.find((img) => img.id === 'ubc-oil-gas'),
    'Project Finance & Guarantees': PlaceHolderImages.find((img) => img.id === 'ubc-finance'),
    'Mining & Metals': PlaceHolderImages.find((img) => img.id === 'ubc-mining'),
    'EPC & Lifecycle Management': PlaceHolderImages.find((img) => img.id === 'ubc-epc'),
  };

  return (
    <div className="space-y-10 relative">
      {/* Decorative Background Elements */}
      <div className="absolute -top-20 -right-20 w-64 h-64 bg-gradient-to-br from-accent/20 to-transparent rounded-full blur-3xl animate-pulse"></div>
      <div className="absolute -bottom-20 -left-20 w-64 h-64 bg-gradient-to-tr from-primary/20 to-transparent rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }}></div>
      
      {/* Title with Animation */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        whileInView={{ opacity: 1, y: 0 }}
        viewport={{ once: true }}
        transition={{ duration: 0.6 }}
        className="text-center relative"
      >
        <div className="inline-flex items-center gap-2 mb-4 px-4 py-2 rounded-full bg-gradient-to-r from-primary/10 via-accent/10 to-primary/10 border border-primary/20">
          <Sparkles className="w-4 h-4 text-accent" />
          <span className="text-sm font-medium text-primary">{badge}</span>
        </div>
        <h3 className="font-headline text-3xl font-bold bg-gradient-to-r from-primary via-accent to-primary bg-clip-text text-transparent">
          {title}
        </h3>
        <div className="mt-4 mx-auto w-24 h-1 bg-gradient-to-r from-transparent via-accent to-transparent rounded-full"></div>
      </motion.div>

      <Carousel
        opts={{
          align: "start",
          loop: true,
        }}
        className="w-full relative"
      >
        <CarouselContent className="-ml-4">
          {items.map((service, index) => {
            const image = ubcImageMap[service.title as keyof typeof ubcImageMap];
            const Icon = ubcIconMap[service.title] || Zap;
            const colorVariant = colorVariants[index % colorVariants.length];
            
            return (
              <CarouselItem key={index} className="pl-4 md:basis-1/2 lg:basis-1/3">
                <motion.div 
                  className="h-full"
                  initial={{ opacity: 0, y: 30 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.5, delay: index * 0.1 }}
                >
                  <Link href="https://ubcholding.co/" target="_blank" rel="noopener noreferrer" className="block h-full">
                    <Card className="group relative flex h-80 flex-col justify-end overflow-hidden rounded-2xl p-6 text-left shadow-xl transition-all duration-500 hover:shadow-2xl hover:-translate-y-2 border-0">
                      {/* Background Image */}
                      {image && (
                        <div className="absolute inset-0">
                          <Image
                            src={image.imageUrl}
                            alt={service.title}
                            fill
                            className="object-cover transition-all duration-700 group-hover:scale-110 group-hover:rotate-2"
                            data-ai-hint={image.imageHint}
                          />
                        </div>
                      )}
                      
                      {/* Gradient Overlays */}
                      <div className="absolute inset-0 bg-gradient-to-t from-black/90 via-black/60 to-black/30 transition-all duration-500 group-hover:from-black/95"></div>
                      <div className={`absolute inset-0 bg-gradient-to-br ${colorVariant} opacity-0 transition-opacity duration-500 group-hover:opacity-100 mix-blend-overlay`}></div>
                      
                      {/* Animated Light Effect */}
                      <div className="absolute inset-0 opacity-0 group-hover:opacity-100 transition-opacity duration-500">
                        <div className="absolute top-0 left-0 w-full h-full bg-gradient-to-br from-white/10 via-transparent to-transparent"></div>
                        <div className="absolute -top-12 -right-12 w-32 h-32 bg-accent/30 rounded-full blur-2xl"></div>
                      </div>

                      {/* Floating Particles */}
                      <div className="absolute inset-0 overflow-hidden opacity-0 group-hover:opacity-100 transition-opacity duration-500">
                        {[...Array(8)].map((_, i) => (
                          <motion.div
                            key={i}
                            className="absolute w-1 h-1 bg-white/60 rounded-full"
                            style={{
                              left: `${20 + i * 10}%`,
                              top: `${30 + (i % 3) * 20}%`,
                            }}
                            animate={{
                              y: [0, -20, 0],
                              opacity: [0.3, 1, 0.3],
                            }}
                            transition={{
                              duration: 2 + Math.random(),
                              repeat: Infinity,
                              delay: i * 0.2,
                            }}
                          />
                        ))}
                      </div>

                      {/* Icon Badge */}
                      <motion.div 
                        className="absolute top-6 right-6 z-20"
                        initial={{ scale: 0, rotate: -180 }}
                        whileInView={{ scale: 1, rotate: 0 }}
                        viewport={{ once: true }}
                        transition={{ duration: 0.5, delay: index * 0.1 + 0.3 }}
                      >
                        <div className="relative">
                          <div className="absolute inset-0 bg-accent/50 rounded-full blur-xl group-hover:blur-2xl transition-all duration-500"></div>
                          <div className="relative w-14 h-14 rounded-full bg-gradient-to-br from-accent/90 to-accent flex items-center justify-center shadow-lg">
                            <Icon className="w-7 h-7 text-white" />
                          </div>
                        </div>
                      </motion.div>

                      {/* Content */}
                      <div className="relative z-10 text-white">
                        <motion.div
                          initial={{ x: -20, opacity: 0 }}
                          whileInView={{ x: 0, opacity: 1 }}
                          viewport={{ once: true }}
                          transition={{ duration: 0.5, delay: index * 0.1 + 0.4 }}
                        >
                          <CardTitle className="font-headline text-xl mb-3 text-white drop-shadow-lg group-hover:text-accent transition-colors duration-300">
                            {service.title}
                          </CardTitle>
                          <div className="flex items-center gap-2 opacity-0 group-hover:opacity-100 transition-all duration-300 transform translate-y-2 group-hover:translate-y-0">
                            <span className="text-sm text-white/90">{learnMore}</span>
                            <ArrowRight className="w-4 h-4 text-accent animate-pulse" />
                          </div>
                        </motion.div>
                      </div>

                      {/* Bottom Shine Effect */}
                      <div className="absolute bottom-0 left-0 right-0 h-1 bg-gradient-to-r from-transparent via-accent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    </Card>
                  </Link>
                </motion.div>
              </CarouselItem>
            );
          })}
        </CarouselContent>
        <CarouselPrevious className="absolute left-[-1.5rem] top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white shadow-xl border-2 border-primary/20 hover:border-accent transition-all duration-300" />
        <CarouselNext className="absolute right-[-1.5rem] top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white shadow-xl border-2 border-primary/20 hover:border-accent transition-all duration-300" />
      </Carousel>
    </div>
  );
}

function NoorServiceList({ title, items, iconMap, badge, reliability, specialized }: { title: string; items: { title: string; description: string }[]; iconMap: { [key: string]: React.ElementType }, badge: string, reliability: string, specialized: string }) {
  const [hoveredIndex, setHoveredIndex] = useState(0);

  return (
    <div className="space-y-10 relative">
      <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[800px] h-[800px] bg-gradient-to-br from-primary/5 via-accent/5 to-transparent rounded-full blur-3xl"></div>
      
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        whileInView={{ opacity: 1, y: 0 }}
        viewport={{ once: true }}
        transition={{ duration: 0.6 }}
        className="text-center relative"
      >
        <div className="inline-flex items-center gap-2 mb-4 px-4 py-2 rounded-full bg-gradient-to-r from-accent/10 via-primary/10 to-accent/10 border border-accent/20">
          <Layers className="w-4 h-4 text-primary" />
          <span className="text-sm font-medium text-primary">{badge}</span>
        </div>
        <h3 className="font-headline text-3xl font-bold bg-gradient-to-r from-accent via-primary to-accent bg-clip-text text-transparent">
          {title}
        </h3>
        <div className="mt-4 mx-auto w-24 h-1 bg-gradient-to-r from-transparent via-primary to-transparent rounded-full"></div>
      </motion.div>

      <div className="grid grid-cols-1 items-start gap-12 lg:grid-cols-12">
        {/* Left Side - Info Panel */}
        <div className="lg:col-span-5 lg:sticky lg:top-24">
          <AnimatePresence mode="wait">
            <motion.div
              key={hoveredIndex}
              initial={{ opacity: 0, y: 20, scale: 0.95 }}
              animate={{ opacity: 1, y: 0, scale: 1 }}
              exit={{ opacity: 0, y: -20, scale: 0.95 }}
              transition={{ duration: 0.4 }}
              className="relative group"
            >
              <motion.div
                className={`absolute -inset-2 bg-gradient-to-r ${noorServiceColors[hoveredIndex % noorServiceColors.length].gradient} rounded-3xl blur-2xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
                animate={{ scale: [1, 1.05, 1] }}
                transition={{ duration: 3, repeat: Infinity }}
              />

              <div className={`relative overflow-hidden border-2 ${noorServiceColors[hoveredIndex % noorServiceColors.length].border} shadow-2xl bg-gradient-to-br from-white via-white ${noorServiceColors[hoveredIndex % noorServiceColors.length].bgGradient} rounded-3xl p-8`}>
                <div className="relative z-10">
                  <motion.div
                    initial={{ scale: 0, rotate: -180 }}
                    animate={{ scale: 1, rotate: 0 }}
                    transition={{ duration: 0.5, type: "spring" }}
                    className="inline-flex mb-6"
                  >
                    <div className="relative">
                      <div className={`relative w-20 h-20 rounded-2xl bg-gradient-to-br ${noorServiceColors[hoveredIndex % noorServiceColors.length].gradient} flex items-center justify-center shadow-xl`}>
                        {(() => {
                          const Icon = iconMap[items[hoveredIndex].title] || Zap;
                          return (
                            <motion.div
                              animate={{ rotate: [0, 5, -5, 0] }}
                              transition={{ duration: 2, repeat: Infinity, repeatDelay: 1 }}
                            >
                              <Icon className="w-10 h-10 text-white" />
                            </motion.div>
                          );
                        })()}
                      </div>
                    </div>
                  </motion.div>
                  <h3 className="text-2xl font-bold mb-4 text-slate-900">
                    {items[hoveredIndex].title}
                  </h3>
                  <p className="text-base text-slate-600 leading-relaxed min-h-[120px]">
                    {items[hoveredIndex].description}
                  </p>
                   <div className="flex items-center justify-between pt-6 border-t-2 border-slate-200/50 mt-6">
                        <div className="flex items-center gap-2">
                           <ShieldCheck className={`w-5 h-5 ${noorServiceColors[hoveredIndex % noorServiceColors.length].accent}`} />
                           <span className="text-sm font-semibold text-slate-600">{reliability}</span>
                        </div>
                        <motion.div
                          animate={{ x: [0, 5, 0] }}
                          transition={{ duration: 1.5, repeat: Infinity }}
                          className={`w-12 h-12 rounded-xl bg-gradient-to-br ${noorServiceColors[hoveredIndex % noorServiceColors.length].gradient} flex items-center justify-center shadow-lg`}
                        >
                          <ArrowRight className="w-6 h-6 text-white" />
                        </motion.div>
                      </div>
                </div>
              </div>
            </motion.div>
          </AnimatePresence>
        </div>

        {/* Right Side - Services Grid */}
        <div className="lg:col-span-7">
          <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
            {items.map((service, index) => {
              const Icon = iconMap[service.title] || Zap;
              const isHovered = hoveredIndex === index;
              const colors = noorServiceColors[index % noorServiceColors.length];

              return (
                <motion.div
                  key={service.title}
                  initial={{ opacity: 0, scale: 0.9 }}
                  whileInView={{ opacity: 1, scale: 1 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.4, delay: index * 0.1 }}
                  onMouseEnter={() => setHoveredIndex(index)}
                  className="relative group"
                >
                  <motion.div
                    className={`absolute -inset-2 bg-gradient-to-r ${colors.gradient} rounded-3xl blur-xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
                    animate={isHovered ? { scale: [1, 1.05, 1] } : {}}
                    transition={{ duration: 2, repeat: Infinity }}
                  />
                  <div
                    className={`relative cursor-pointer bg-white/80 backdrop-blur-sm p-6 rounded-3xl border-2 ${isHovered ? colors.border : 'border-slate-200/50'} shadow-xl hover:shadow-2xl transition-all duration-500 overflow-hidden group-hover:-translate-y-2`}
                  >
                     <motion.div
                          className={`absolute inset-0 bg-gradient-to-br ${colors.bgGradient} opacity-0 group-hover:opacity-100 transition-opacity duration-700`}
                     />
                    <div className="relative z-10 flex items-start gap-4">
                      <motion.div
                        animate={{ scale: isHovered ? 1.1 : 1 }}
                        transition={{ duration: 0.5 }}
                        className="relative flex-shrink-0"
                      >
                        <div className={`relative flex h-16 w-16 items-center justify-center rounded-2xl bg-gradient-to-br ${isHovered ? colors.gradient : 'from-slate-100 to-slate-200'} shadow-lg transition-all duration-500`}>
                          <Icon className={`h-8 w-8 ${isHovered ? 'text-white' : 'text-slate-700'} transition-all duration-500`} />
                        </div>
                      </motion.div>
                      <div className="flex-1 min-w-0">
                        <h3 className={`text-lg font-bold mb-2 transition-colors duration-300 ${isHovered ? colors.accent : 'text-slate-900'}`}>
                          {service.title}
                        </h3>
                         <motion.div
                              initial={{ opacity: 0, y: 10 }}
                              animate={{ opacity: isHovered ? 1 : 0, y: isHovered ? 0 : 10 }}
                              transition={{ duration: 0.3 }}
                              className={`inline-flex items-center gap-1.5 px-3 py-1 rounded-lg bg-gradient-to-r ${colors.gradient} text-white text-xs font-semibold shadow-md`}
                            >
                              <Sparkles className="w-3 h-3" />
                              <span>{specialized}</span>
                            </motion.div>
                      </div>
                    </div>
                     <motion.div
                        className={`absolute bottom-0 left-0 right-0 h-1 bg-gradient-to-r ${colors.gradient}`}
                        initial={{ scaleX: 0 }}
                        animate={{ scaleX: isHovered ? 1 : 0 }}
                        transition={{ duration: 0.5 }}
                        style={{ transformOrigin: 'left' }}
                      />
                  </div>
                </motion.div>
              );
            })}
          </div>
        </div>
      </div>
    </div>
  );
}

export function ServicesOverview() {
  const { content } = useLanguage();
  const { services } = content;

  return (
    <section id="services" className="scroll-mt-20 relative overflow-hidden bg-gradient-to-b from-white via-blue-50/20 to-white py-16">
      {/* Background Decorations */}
      <div className="absolute top-0 left-0 w-full h-full pointer-events-none overflow-hidden">
        <div className="absolute top-20 -left-40 w-96 h-96 bg-gradient-to-br from-primary/10 to-transparent rounded-full blur-3xl"></div>
        <div className="absolute bottom-20 -right-40 w-96 h-96 bg-gradient-to-tl from-accent/10 to-transparent rounded-full blur-3xl"></div>
      </div>

      <div className="mx-auto max-w-2xl text-center relative z-10 mb-16">
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
        >
          {/* Badge */}
          <div className="inline-flex items-center gap-2 mb-6 px-6 py-3 rounded-full bg-gradient-to-r from-primary/10 via-accent/10 to-primary/10 border border-primary/20 shadow-lg">
            <Wrench className="w-5 h-5 text-accent animate-pulse" />
            <span className="text-sm font-semibold text-primary">{services.badge}</span>
          </div>

          {/* Main Title */}
          <h2 className="font-headline text-4xl font-bold tracking-tight sm:text-5xl mb-4">
            <span className="bg-gradient-to-r from-primary via-accent to-primary bg-clip-text text-transparent">
              {services.title}
            </span>
          </h2>

          {/* Subtitle */}
          <p className="mt-4 text-lg text-muted-foreground max-w-xl mx-auto">
            {services.subtitle}
          </p>

          {/* Decorative Line */}
          <div className="mt-8 mx-auto w-32 h-1.5 bg-gradient-to-r from-transparent via-accent to-transparent shadow-lg shadow-accent/50"></div>
        </motion.div>
      </div>

      <div className="container mx-auto mt-20 grid grid-cols-1 gap-24 lg:grid-cols-1 relative z-10">
        <ServiceCarousel 
            title={services.ubc.title} 
            items={services.ubc.items} 
            badge={services.ubc.badge}
            learnMore={services.ubc.learnMore}
        />
        <NoorServiceList 
            title={services.noor.title} 
            items={services.noor.items} 
            iconMap={noorIconMap}
            badge={services.noor.badge}
            reliability={services.noor.reliability}
            specialized={services.noor.specialized}
        />
      </div>
    </section>
  );
}
